/*
 * drivers/char/genrtc.c -- generic dummy /dev/rtc
 *
 * started 11/12/1999 Sam Creasey
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive
 * for more details.
 * 
 * portions of this code based on rtc.c, which is
 * Copyright (C) 1996 Paul Gortmaker.  See that file for more information.
 */


#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/miscdevice.h>
#include <linux/malloc.h>
#include <linux/fcntl.h>
//#include <linux/mc146818rtc.h>
#include <linux/rtc.h>
#include <linux/init.h>
#include <linux/kd.h>


#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/system.h>
#include <asm/machdep.h>

#define GEN_RTC_VERSION "v1.01 11/17/1999"

/* make sure only one process opens /dev/rtc */
static unsigned char gen_rtc_opened = 0;

/* uie emulation -- basically a 1 second timer */
static unsigned char gen_uie_on = 0;
static unsigned char gen_uie_data = 0;
static struct timer_list gen_uie_timer;
static DECLARE_WAIT_QUEUE_HEAD(gen_rtc_wait);

/* handle timer hit -- basically acts like we got a uie */
static void gen_do_uie_timer(unsigned long t)
{
	gen_uie_data = 1;
	wake_up_interruptible(&gen_rtc_wait);
	mod_timer(&gen_uie_timer, jiffies + HZ);

}
	

/* file operations, some derived from rtc.c */

static loff_t gen_rtc_llseek(struct file *file, loff_t offset, int origin)
{
        return -ESPIPE;
}

/* just like the normal rtc... */

static ssize_t gen_rtc_read(struct file *file, char *buf,
                        size_t count, loff_t *ppos)
{
        DECLARE_WAITQUEUE(wait, current);
        unsigned long data;
        ssize_t retval;

        if (count < sizeof(unsigned long))
                return -EINVAL;

        add_wait_queue(&gen_rtc_wait, &wait);

        current->state = TASK_INTERRUPTIBLE;

        while ((data = xchg(&gen_uie_data, 0)) == 0) {
                if (file->f_flags & O_NONBLOCK) {
                        retval = -EAGAIN;
                        goto out;
                }
                if (signal_pending(current)) {
                        retval = -ERESTARTSYS;
                        goto out;
                }
                schedule();
        }
        retval = put_user(data, (unsigned long *)buf);
        if (!retval)
                retval = sizeof(unsigned long);
 out:
        current->state = TASK_RUNNING;
        remove_wait_queue(&gen_rtc_wait, &wait);

        return retval;
}

/* handle ioctls -- many of them are not implemented */

static int gen_rtc_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
                     unsigned long arg)
{
	struct rtc_time wtime;
	struct hwclk_time hwtime;

	switch(cmd) {
	case RTC_RD_TIME:
		mach_hwclk(0, &hwtime);
		wtime.tm_sec = hwtime.sec;
		wtime.tm_min = hwtime.min;
		wtime.tm_hour = hwtime.hour;
		wtime.tm_mday = hwtime.day;
		wtime.tm_mon = hwtime.mon;
		wtime.tm_year = hwtime.year;
		/* ignore wday, yday, isdst */
		return copy_to_user((void *)arg, &wtime, sizeof(wtime)) ? -EFAULT : 0;
	
	case RTC_SET_TIME:
		if(copy_from_user(&wtime, (struct rtc_time *)arg, 
				  sizeof(struct rtc_time)))
			return -EFAULT;

		hwtime.sec = wtime.tm_sec;
		hwtime.min = wtime.tm_min;
		hwtime.hour = wtime.tm_hour;
		hwtime.day = wtime.tm_mday;
		hwtime.mon = wtime.tm_mon;
		hwtime.year = wtime.tm_year;

		mach_hwclk(1, &hwtime);
		return 0;

	case RTC_UIE_ON:
		/* set a timer to fake clock ticks every second */
		if(gen_uie_on) 
			return 0;

		gen_uie_on = 1;
		gen_uie_timer.expires = jiffies + HZ;
		add_timer(&gen_uie_timer);
		
		return 0;

	case RTC_UIE_OFF:
		if(!gen_uie_on)
			return 0;
		
		gen_uie_on = 0;
		del_timer(&gen_uie_timer);
		return 0;
		
	default:
		break;
	}

	return -EINVAL;
}

static int gen_rtc_open(struct inode *inode, struct file *file)
{
	if(gen_rtc_opened)
		return -EBUSY;

	gen_uie_data = 0;
	gen_rtc_opened = 1;
	return 0;
}

static int gen_rtc_release(struct inode *inode, struct file *file)
{
	if(gen_uie_on) {
		gen_uie_on = 0;
		del_timer(&gen_uie_timer);
	}

	gen_rtc_opened = 0;
	
	return 0;
}
		
/*
 *      The various file operations we support.
 */

static struct file_operations gen_rtc_fops = {
	THIS_MODULE,
        gen_rtc_llseek,
        gen_rtc_read,
        NULL,           /* No write */
        NULL,           /* No readdir */
	NULL,           /* No poll -- needed? */
        gen_rtc_ioctl,
        NULL,           /* No mmap */
        gen_rtc_open,
        NULL,           /* flush */
        gen_rtc_release
};

/* misc device stuff */

static struct miscdevice gen_rtc_dev=
{
        RTC_MINOR,
        "rtc",
        &gen_rtc_fops
};

int __init rtc_generic_init(void)
{
	
	printk("Generic RTC Driver %s Sam Creasey (sammy@oh.verio.com)\n", GEN_RTC_VERSION);

	init_timer(&gen_uie_timer);
	gen_uie_timer.function = gen_do_uie_timer;

	misc_register(&gen_rtc_dev);


	return 0;
}
	
