/*
 * This code is (C) 1995 Jes Sorensen (jds@kom.auc.dk).
 *
 * This is a driver for the GVP IO-Extender (duart, midi, parallel
 * board) for the Amiga (so far only the serial part is implemented).
 *
 * The code is partly based on Roman's Atari drivers, Hamish'
 * driver the internal Amiga port and the PC drivers.
 *
 * The board info was kindly supplied by Erick Quackenbush, former GVP
 * employee - thanks. The missing info on about the "secret" register
 * was obtained from the GVP SCSI driver though.
 *
 * Thanks to Niels Pontoppidan (neil@dannug.dk), for letting me borrow
 * his IO-Extender, and thus making it possible for me to do this
 * driver. Also thanks to Ralph Babel and Roman Hodek for a few hints
 * and to Michael Goodwin for doing all the beta-testing - especially
 * after I returned the board I borrowed from Niels.
 *
 * If you can use this code (or parts of it) - feel free to copy it,
 * but please let me know what parts you can use and where you use it.
 *
 * As the code probably isn't bug-free yet, I would also like to know
 * if you discover any bugs.
 *
 * Facts: 
 *
 * The tty-assignment is done dynamically so only the needed number of
 * tty's is allocated, leaving room for other serial boards in the
 * system. This needed some changes in serial.c aswell as the other
 * drivers, but I do find this the best way to handle things - I hate
 * static allocation!
 *
 * Note: To use the serial ports at rates higher than 38400, you can
 * compile the driver with IOEXT_FAST_RATES = 1, and then select
 * one of these rates:
 *
 * 300 = 57600, 1200 = 115200
 *
 * The driver should be able to handle multiple IO-Extenders, but I
 * haven't been able to test this as I only got access to one board at
 * the moment. The number of extenders is limited to five (same as the
 * maximum number of Zorro slots) but if you wan't to use more than
 * two you should change "NR_PORTS" in serial.h to fit your needs.
 *
 * 07/13/95: Initial version - the code is definately _not_ perfect
 *           yet - it doesn't even work :-(
 * 10/01/95: First version, it seems to work pretty well, even at
 *           serial-rates higher than 19200.
 *           There is still one major problem, as I get lotsa overruns
 *           when the harddisk is accessed during receives. This is
 *           probably caused by the SCSI-driver running too long with
 *           interrupts off, so I'm not sure there is anything I can
 *           do about it in here.
 * 10/04/95: Added support for higher (57k6 and 115k2) baud-rates when
 *           the ASYNC_SPD_{HI,VHI} flags are set.
 * 95/10/19: Adapted to 1.2 (Andreas Schwab).
 *	     Compiles, but otherwise untestet!
 * 95/11/26: Updated for 1.2.13pl3 - updated the GVP detection aswell.
 * 96/01/17: Hopefully solved the GVP Zorro detection problem in
 *           1.2.13pl4 that occoured when a user had several GVP
 *           boards installed. (Jes)
 * 96/02/07: The driver actually decreases info->xmit_cnt-- now
 *           whenever a character is transmitted, so we don't just
 *           transmit the same char now. This is the reason why this
 *           driver hasn't worked under 1.2 so far. Thanks to Murray
 *           Bennett for this, found it in his patches.
 * 96/02/27: Integrate various changes to make it work. (murray)
 *           No need to protect against interrupt reentrance.
 *           No need to acknowledge the interrupt (it is done for us).
 *           Process as many conditions as possible during the interrupt.
 *           Use the 16 byte transmit fifo.
 *           Handle THRE interrupts properly. This is a level driven interrupt,
 *           not an edge triggered interrupt, so only enable it when we have
 *           characters to send, and disable it as soon as we have sent the
 *           last character.
 *           Don't try to clear pending interrupts during ioext_init() since
 *           reading IIR achieves nothing.
 *           Register a separate interrupt handler per board. For some
 *           strange reason, I couldn't get it to work with a single handler.
 * 97/01/14: Add module support. <msteveb@ozemail.com.au>
 *           Allow multiple IOExtenders to be used (this was broken).
 *           Perhaps add support for the serial port on the GForce-040.
 * 97/03/29: Add printer and PLIP support <msteveb@ozemail.com.au>
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/interrupt.h>
#include <linux/malloc.h>
#include <linux/termios.h>
#include <linux/tty.h>
#include <linux/m68kserial.h>
#include <linux/netdevice.h>
#include <linux/lp.h>
#include <linux/16c552.h>
#include <linux/init.h>

#include <asm/setup.h>
#include <asm/irq.h>
#include <asm/amigahw.h>
#include <asm/amigaints.h>
#include <linux/zorro.h>

#include "ioext.h"

/* Set these to 0 when the driver is finished */
#define IOEXT_DEBUG 0
#define DEBUG_IRQ 0
#define DEBUG 0
#define DEBUG_FLOW 0

#define FIFO_SIZE 16  /* Size of hardware send and receive FIFOs */

#define IOEXT_FAST_RATES 1   /*
			      * When this is set, the rates 110 and
			      * 134 will run 57600 and 115200
			      * respectively.
			      */

#define FIFO_TRIGGER_LEVEL FIFO_TRIG_8
                             /*
			      * There are 4 receiver FIFO-interrupt *
			      * trigger levels (FIFO_TRIG_x), that  *
			      * indicates how many bytes are to be  *
			      * allowed in the receiver-FIFO before *
			      * an interrupt is generated:          *
			      *                x =  1 =  1 byte	    *
			      *                x =  4 =  4 bytes    *
			      *                x =  8 =  8 bytes    *
			      *                x = 14 = 14 bytes    *
			      * 14 works for me, but if you keep    *
			      * getting overruns try lowering this  *
			      * value one step at a time.           *
			      */

/***************************** Prototypes *****************************/
static void ioext_ser_interrupt(volatile struct uart_16c550 *uart, int line, int *spurious_count);
static void ser_init( struct m68k_async_struct *info );
static void ser_deinit( struct m68k_async_struct *info, int leave_dtr );
static void ser_enab_tx_int( struct m68k_async_struct *info, int enab_flag );
static int  ser_check_custom_divisor(struct m68k_async_struct *info,
				     int baud_base, int divisor);
static void ser_change_speed( struct m68k_async_struct *info );
static void ser_throttle( struct m68k_async_struct *info, int status );
static void ser_set_break( struct m68k_async_struct *info, int break_flag );
static void ser_get_serial_info( struct m68k_async_struct *info,
				struct serial_struct *retinfo );
static unsigned int ser_get_modem_info( struct m68k_async_struct *info );
static int ser_set_modem_info( struct m68k_async_struct *info, int new_dtr,
			      int new_rts );
static void ser_stop_receive(struct m68k_async_struct *info);
static int ser_trans_empty(struct m68k_async_struct *info);
/************************* End of Prototypes **************************/

/*
 * SERIALSWITCH structure for the GVP IO-Extender serial-board.
 */

static SERIALSWITCH ioext_ser_switch = {
    ser_init,
    ser_deinit,
    ser_enab_tx_int,
    ser_check_custom_divisor,
    ser_change_speed,
    ser_throttle,
    ser_set_break,
    ser_get_serial_info,
    ser_get_modem_info,
    ser_set_modem_info,
    NULL,
    ser_stop_receive, ser_trans_empty, NULL
};

static int ioext_baud_table[18] = {
 	/* B0     */ 0, /* Newer use this value !!! */
	/* B50    */ 9216,
	/* B75    */ 6144,
	/* B110   */ 4189, /* There's a little rounding on this one */
	/* B134	  */ 3439, /* Same here! */
	/* B150	  */ 3072,
	/* B200	  */ 2304,
#if IOEXT_FAST_RATES
	/* B57600 */ 8,
#else
	/* B300	  */ 1536,
#endif
	/* B600	  */ 768,
#if IOEXT_FAST_RATES
	/* B115k2 */ 4,
#else
	/* B1200  */ 384,
#endif
	/* B1800  */ 256,
	/* B2400  */ 192,
	/* B4800  */ 96,
	/* B9600  */ 48,
	/* B19200 */ 24,
	/* B38400 */ 12,   /* The last of the standard rates.  */
	/* B57600 */ 8,    /* ASYNC_SPD_HI                     */
	/* B115K2 */ 4     /* ASYNC_SPD_VHI                    */
};

int ioext_num;
IOExtInfoType ioext_info[MAX_IOEXT];

/*
 * Functions
 *
 * dev_id = ioext_info.
 *
 */
static void ioext_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
  int i, b;

  if (ioext_num == 0) {
    /* This interrupt can't be for us */
    return;
  }

  for (b = 0; b < ioext_num; b++) {
    IOEXT_struct *board = ioext_info[b].board;
    IOExtInfoType *board_info = &ioext_info[b];

    /* Check if any irq is pending for the current board.  */
    /* Note: The IRQ_PEND-bit=0 if an irq is pending.      */

    if (!(board->CNTR & GVP_IRQ_PEND)) {
      /* DON'T print anything here */
      continue;
    }

    /* Service any uart irqs. */
    for (i = 0; i < board_info->num_uarts; i++) {
      ioext_ser_interrupt(board_info->uart[i], board_info->line[i], &board_info->spurious_count);
    }

#if defined(CONFIG_GVPIOEXT_PLIP) || defined(CONFIG_GVPIOEXT_PLIP_MODULE)
    if (board_info->par_use & IOEXT_PAR_PLIP) {
      ioext_plip_interrupt(board_info->dev, &board_info->spurious_count);
      /*printk("Called ioext_plip_interrupt\n");*/
    }
    else
#endif
#if defined(CONFIG_GVPIOEXT_LP) || defined(CONFIG_GVPIOEXT_LP_MODULE)
    if (board_info->par_use & IOEXT_PAR_LP) {
      ioext_lp_interrupt(board_info->lp_dev, &board_info->spurious_count);
      /*printk("Called ioext_lp_interrupt\n");*/
    }
    else
#endif
    if (!board_info->par_use) {
      /* Ack parallel port interrupt */
      (void)board->par.STATUS;
      /*printk("ioext_ser_interrupt() handled par interrupt\n");*/
    }

    if (board_info->spurious_count > 10000) {
      board->CNTR &= ~GVP_IRQ_ENA;
      printk("Too many spurious interrupts, disabling board irq (CTRL=%02X)\n", board->par.CTRL);
      board_info->spurious_count = 0;
    }
  } /* for b */
}

/*
** ioext_ser_interrupt()
** 
** Check for and handle interrupts for this uart.
*/
static void ioext_ser_interrupt(volatile struct uart_16c550 *uart, int line, int *spurious_count)
{
  struct m68k_async_struct *info = rs_table + line;

  u_char iir;
  u_char lsr;
  int ch;

  iir = uart->IIR;

  ++*spurious_count;

#if DEBUG_IRQ
  printk("ttyS%d: IER=%02X, IIR=%02X, LSR=%02X, MSR=%02X\n", line, uart->IER, iir, uart->LSR, uart->MSR);
#endif

  while (!(iir & IRQ_PEND)) {
    /* IRQ for this uart */
#if DEBUG_IRQ
    printk("IRQ_PEND on ttyS%d...\n", line);
#endif

    /* This really is our interrupt */
    *spurious_count = 0;

    switch (iir & (IRQ_ID1 | IRQ_ID2 | IRQ_ID3)) {
      case IRQ_RLS: /* Receiver Line Status */
#if DEBUG_FLOW
      printk("RLS irq on ttyS%d\n", line);
#endif
      case IRQ_CTI: /* Character Timeout */
      case IRQ_RDA: /* Received Data Available */
#if DEBUG_IRQ
      printk("irq (IIR=%02X) on ttyS%d\n", line, iir);
#endif
      /*
       * Copy chars to the tty-queue ...
       * Be careful that we aren't passing one of the
       * Receiver Line Status interrupt-conditions without noticing.
       */
      {
        int got = 0;

        lsr = uart->LSR;
#if DEBUG_IRQ
        printk("uart->LSR & DR = %02X\n", lsr & DR);
#endif
        while (lsr & DR) {
          u_char err = 0;
          ch = uart->RBR;
#if DEBUG_IRQ
          printk("Read a char from the uart: %02X, lsr=%02X\n", ch, lsr);
#endif
          if (lsr & BI) {
            err = TTY_BREAK;
          }
          else if (lsr & PE) {
            err = TTY_PARITY;
          }
          else if (lsr & OE) {
            err = TTY_OVERRUN;
          }
          else if (lsr & FE) {
            err = TTY_FRAME;
          }
#if DEBUG_IRQ
          printk("rs_receive_char(ch=%02X, err=%02X)\n", ch, err);
#endif
          rs_receive_char(info, ch, err);
          got++;
          lsr = uart->LSR;
        }
#if DEBUG_FLOW
        printk("[%d<]", got);
#endif
      }
      break;

    case IRQ_THRE: /* Transmitter holding register empty */
      {
        int fifo_space = 16;
        int sent = 0;

#if DEBUG_IRQ
        printk("THRE-irq for ttyS%d\n", line);
#endif

        /* If the uart is ready to receive data and there are chars in */
        /* the queue we transfer all we can to the uart's FIFO         */
        if (info->xmit_cnt <= 0 || info->tty->stopped ||
            info->tty->hw_stopped) {
          /* Disable transmitter empty interrupt */
          uart->IER &= ~(ETHREI);
          /* Need to send a char to acknowledge the interrupt */
          uart->THR = 0;
#if DEBUG_FLOW
          if (info->tty->hw_stopped) {
            printk("[-]");
          }
          if (info->tty->stopped) {
            printk("[*]");
          }
#endif
          break;
        }

        /* Handle software flow control */
        if (info->x_char) {
#if DEBUG_FLOW
          printk("[^%c]", info->x_char + '@');
#endif
          uart->THR = info->x_char;
          info->x_char = 0;
          fifo_space--;
          sent++;
        }

        /* Fill the fifo */
        while (fifo_space > 0) {
          fifo_space--;
#if DEBUG_IRQ
          printk("Sending %02x to the uart.\n", info->xmit_buf[info->xmit_tail]);
#endif
          uart->THR = info->xmit_buf[info->xmit_tail++];
          sent++;
          info->xmit_tail = info->xmit_tail & (SERIAL_XMIT_SIZE-1);
          if (--info->xmit_cnt == 0) {
            break;
          }
        }
#if DEBUG_FLOW
          printk("[>%d]", sent);
#endif

        if (info->xmit_cnt == 0) {
#if DEBUG_IRQ
          printk("Sent last char - turning off THRE interrupts\n");
#endif
          /* Don't need THR interrupts any more */
          uart->IER &= ~(ETHREI);
        }

        if (info->xmit_cnt < WAKEUP_CHARS) {
          rs_sched_event(info, RS_EVENT_WRITE_WAKEUP);
        }
      }
      break;

    case IRQ_MS: /* Must be modem status register interrupt? */
      {
        u_char msr = uart->MSR;
#if DEBUG_IRQ
        printk("MS-irq for ttyS%d: %02x\n", line, msr);
#endif

        if (info->flags & ASYNC_INITIALIZED) {
          if (msr & DCTS) {
            rs_check_cts(info, (msr & CTS)); /* active high */
#if DEBUG_FLOW
            printk("[%c-%d]", (msr & CTS) ? '^' : 'v', info->tty ? info->tty->hw_stopped : -1);
#endif
          }
          if (msr & DDCD) {
#if DEBUG
           printk("rs_dcd_changed(%d)\n", !(msr & DCD));
#endif
            rs_dcd_changed(info, !(msr & DCD)); /* active low */
          }
        }
      }
      break;

    default:
#if DEBUG_IRQ
        printk("Unexpected irq for ttyS%d\n", line);
#endif
      break;
    } /* switch (iir) */
    iir = uart->IIR;
  } /* while IRQ_PEND */
}

static void ser_init(struct m68k_async_struct *info)
{
#if DEBUG
	printk("ser_init\n");
#endif

	while ((curruart(info)->LSR) & DR) {
#if IOEXT_DEBUG
		printk("Emptying uart\n");
#endif
		(void)curruart(info)->RBR;
	}

	/* Set DTR and RTS */
	curruart(info)->MCR |= (DTR | RTS | OUT2);

  /* Enable interrupts. IF_EXTER irq has already been enabled in ioext_init()*/
  /* DON'T enable ETHREI here because there is nothing to send yet (murray) */
	curruart(info)->IER |= (ERDAI | ELSI | EMSI);

	MOD_INC_USE_COUNT;
}


static void ser_deinit(struct m68k_async_struct *info, int leave_dtr)
{
#if DEBUG
	printk("ser_deinit\n");
#endif

	/* Wait for the uart to get empty */
	while(!(curruart(info)->LSR & TEMT)) {
		mb();
#if IOEXT_DEBUG
		printk("Waiting for the transmitter to finish\n");
#endif
	}

	while(curruart(info)->LSR & DR) {
#if IOEXT_DEBUG
		printk("Uart not empty - flushing!\n");
#endif
		(void)curruart(info)->RBR;
	}

	/* No need to disable UART interrupts since this will already
	 * have been done via ser_enab_tx_int() and ser_stop_receive()
	 */

	ser_RTSoff(info);
	if (!leave_dtr) {
		ser_DTRoff(info);
	}

	MOD_DEC_USE_COUNT;
}

/*
** ser_enab_tx_int()
** 
** Enable or disable tx interrupts.
** Note that contrary to popular belief, it is not necessary to
** send a character to cause an interrupt to occur. Whenever the
** THR is empty and THRE interrupts are enabled, an interrupt will occur.
** (murray)
*/
static void ser_enab_tx_int(struct m68k_async_struct *info, int enab_flag)
{
	if (enab_flag) {
		curruart(info)->IER |= ETHREI;
	}
	else {
		curruart(info)->IER &= ~(ETHREI);
	}
}

static int  ser_check_custom_divisor(struct m68k_async_struct *info,
				     int baud_base, int divisor)
{
	/* Always return 0 or else setserial spd_hi/spd_vhi doesn't work */
	return 0;
}

static void ser_change_speed(struct m68k_async_struct *info)
{
	unsigned int cflag, baud, chsize, stopb, parity, aflags;
	unsigned int div = 0, ctrl = 0;

#if DEBUG
	printk("ser_change_speed\n");
#endif

	if (!info->tty || !info->tty->termios) return;

	cflag  = info->tty->termios->c_cflag;
	baud   = cflag & CBAUD;
	chsize = cflag & CSIZE;
	stopb  = cflag & CSTOPB;
	parity = cflag & (PARENB | PARODD);
	aflags = info->flags & ASYNC_SPD_MASK;

	if (cflag & CRTSCTS)
		info->flags |= ASYNC_CTS_FLOW;
	else
		info->flags &= ~ASYNC_CTS_FLOW;
	if (cflag & CLOCAL)
		info->flags &= ~ASYNC_CHECK_CD;
	else
		info->flags |= ASYNC_CHECK_CD;

#if DEBUG
	printk("Changing to baud-rate %i\n", baud);
#endif

	if (baud & CBAUDEX) {
		baud &= ~CBAUDEX;
		if (baud < 1 || baud > 4)
			info->tty->termios->c_cflag &= ~CBAUDEX;
		else
			baud += 15;
	}
	if (baud == 15){
		switch (aflags) {
		case ASYNC_SPD_HI:   /*  57k6 */
			baud += 1;
			break;
		case ASYNC_SPD_VHI:  /* 115k2 */
			baud += 2;
			break;
		case ASYNC_SPD_SHI:  /* 230k4 */
			baud += 3;
			break;
		case ASYNC_SPD_WARP:  /* 460k8 */
			baud += 4;
			break;
		case ASYNC_SPD_CUST:
			div = info->custom_divisor;
			break;
		}
	}
	if (!div){
		/* Maximum speed is 115200 */
		if (baud > 17) baud = 17;
		div = ioext_baud_table[baud];
	}

	if (!div) {
		/* speed == 0 -> drop DTR */
#if DEBUG
		printk("Dropping DTR\n");
#endif
		ser_DTRoff(info);
		return;
	}

  /*
   * We have to set DTR when a valid rate is chosen, otherwise DTR
   * might get lost when programs use this sequence to clear the line:
   *
   * change_speed(baud = B0);
   * sleep(1);
   * change_speed(baud = Bx); x != 0
   *
   * The pc-guys do this aswell.
   */
	ser_DTRon(info);

	if (chsize == CS8) {
#if DEBUG
		printk("Setting serial word-length to 8-bits\n");
#endif
		ctrl |= data_8bit;
	}
	else if (chsize == CS7) {
#if DEBUG
		printk("Setting serial word-length to 7-bits\n");
#endif
		ctrl |= data_7bit;
	}
	else if (chsize == CS6) {
#if DEBUG
		printk("Setting serial word-length to 6-bits\n");
#endif
		ctrl |= data_6bit;
	}
	else if (chsize == CS5) {
#if DEBUG
		printk("Setting serial word-length to 5-bits\n");
#endif
		ctrl |= data_5bit;
	};


	/* If stopb is true we set STB which means 2 stop-bits - */
	/* otherwise we  only get 1 stop-bit.                    */
	ctrl |= (stopb ? STB : 0);
     
	ctrl |= ((parity & PARENB) ? ((parity & PARODD) ? (PEN) : (PEN |
								   EPS)) :
		 0x00 ); 

#if DEBUG
	printk ("Storing serial-divisor %i\n", div);
#endif

	curruart(info)->LCR = (ctrl | DLAB);

	/* Store high byte of divisor */
	curruart(info)->DLM = ((div >> 8) & 0xff);
  
	/* Store low byte of divisor */

	curruart(info)->DLL = (div & 0xff);

	curruart(info)->LCR = ctrl;
}


static void ser_throttle(struct m68k_async_struct *info, int status){

#if DEBUG
	printk("ser_throttle\n");
#endif
	if (status){
		ser_RTSoff(info);
	}
	else{
		ser_RTSon(info);
	}
}


static void ser_set_break(struct m68k_async_struct *info, int break_flag)
{
#if IOEXT_DEBUG
	printk("ser_set_break\n");
#endif
	if (break_flag)
		curruart(info)->LCR |= SET_BREAK;
	else
		curruart(info)->LCR &= ~SET_BREAK;
}


static void ser_get_serial_info(struct m68k_async_struct *info,
				struct serial_struct *retinfo)
{
#if DEBUG
	printk("ser_get_serial_info\n");
#endif

	retinfo->baud_base = IOEXT_BAUD_BASE;
	retinfo->xmit_fifo_size = FIFO_SIZE; /* This field is currently      */
					     /* ignored, by the upper layers */
					     /* layers of the serial-driver. */
	retinfo->custom_divisor = info->custom_divisor;

}

static unsigned int ser_get_modem_info(struct m68k_async_struct *info)
{
	unsigned char msr, mcr;

#if DEBUG
	printk("ser_get_modem_info\n");
#endif

	msr = curruart(info)->MSR;
	mcr = curruart(info)->MCR; /* The DTR and RTS are located in the */
				   /* ModemControlRegister ...           */

	return(
		((mcr & DTR) ? TIOCM_DTR : 0) |
		((mcr & RTS) ? TIOCM_RTS : 0) |

		((msr & DCD) ? 0 : TIOCM_CAR) | /* DCD is active low */
		((msr & CTS) ? TIOCM_CTS : 0) |
		((msr & DSR) ? TIOCM_DSR : 0) |
		((msr & RING_I) ? TIOCM_RNG : 0)
		);
}

static int ser_set_modem_info(struct m68k_async_struct *info, int new_dtr,
			      int new_rts)
{
#if DEBUG
	printk("ser_set_modem_info new_dtr=%i new_rts=%i\n", new_dtr, new_rts);
#endif
	if (new_dtr == 0)
		ser_DTRoff(info);
	else if (new_dtr == 1)
		ser_DTRon(info);

	if (new_rts == 0)
		ser_RTSoff(info);
	else {
		if (new_rts == 1)
			ser_RTSon(info);
	}

	return 0;
}

static void ser_stop_receive (struct m68k_async_struct *info)
{
	/* Disable uart receive and status interrupts */
	curruart(info)->IER &= ~(ERDAI | ELSI | EMSI);
}

static int ser_trans_empty (struct m68k_async_struct *info)
{
	return (curruart(info)->LSR & THRE);
}

/*
** init_ioext_uart()
** 
** 
*/
static void init_ioext_uart(struct uart_16c550 *uart)
{
    /* Wait for the uart to get empty */
    int count = 32;
    while (!(uart->LSR & TEMT) && (count-- != 0)) {
	    mb();
#if IOEXT_DEBUG
	    printk("Waiting for transmitter to finish\n");
#endif
    }

    /*
     * Set the uart to a default setting of 8N1 - 9600
     */

    uart->LCR = (data_8bit | DLAB);
    uart->DLM = 0;
    uart->DLL = 48;
    uart->LCR = (data_8bit);

    /*
     * Enable + reset the tx and rx FIFO's.
     * Set the rx FIFO-trigger count.
     */

    uart->FCR =  (FIFO_ENA | RCVR_FIFO_RES |
		  XMIT_FIFO_RES | FIFO_TRIGGER_LEVEL);

    /*
     * Disable all uart interrups (they will be re-enabled in
     * ser_init when they are needed).
     */
    uart->IER = 0;

    /*
     * Master interrupt enable - National semconductors doesn't like
     * to follow standards, so their version of the chip is
     * different and I only had a copy of their data-sheets :-(
     */
    uart->MCR = OUT2;
}

/*
 * Detect and initialize all IO-Extenders in this system.
 *
 * Both PLIP/parallel and serial devices are configured.
 */
int __init ioext_init(void)
{
	struct zorro_dev *z = NULL;
	int isr_installed = 0, is_ioext;
	static char support_string[50] = "io-extender serial";

	if (!MACH_IS_AMIGA) {
		return -ENODEV;
	}

	memset(ioext_info, 0, sizeof(ioext_info));

	while ((z = zorro_find_device(ZORRO_WILDCARD, z))) {
		const char *name;
		caddr_t address;
		IOEXT_struct *board;
		int num_uarts;
		int i;

		if (z->id == ZORRO_PROD_GVP_IO_EXTENDER) {
			is_ioext = 1;
			name = "IO-Extender Multi I/O";
		} else if (z->id == ZORRO_PROD_GVP_GFORCE_040_1) {
			is_ioext = 0;
			name = "GForce 040 Accelerator";
		} else
			continue;

		address = z->resource.start;
		if (!request_mem_region((unsigned long)address,
			    		sizeof(IOEXT_struct), "16c552 DACE"))
			continue;
		strcpy(z->name, name);

		/* Wait for the board to be ready and disable everything */
		board = (IOEXT_struct *) ZTWO_VADDR(address);

		ioext_info[ioext_num].board = board;

		while (!board->CNTR & GVP_IRQ_PEND) {
			printk("GVP-irq pending \n");
		}
      
		/*
		 * Disable board-interrupts for the moment.
		 */
		board->CNTR &= ~GVP_IRQ_ENA;

		/* IO Extender has a second serial port. GForce 040 does not */
		num_uarts = is_ioext ? 2 : 1;

		/*
		 * Register the serial port devices.
		 */
		for (i = 0; i < num_uarts; i++) {
			struct uart_16c550 *uart = (i == 0) ? &board->uart0
				: &board->uart1;
			struct serial_struct req;
			int line;

			req.line = -1; /* first free ttyS? device */
			req.type = SER_IOEXT;
			req.port = (int)uart;
			if ((line = register_serial(&req)) < 0) {
				printk( "Cannot register IO-Extender serial port: no free device\n" );
				break;
			}

			/* Add this uart to our ioext_info[] table */
			ioext_info[ioext_num].line[ioext_info[ioext_num].num_uarts] = line;
			ioext_info[ioext_num].uart[ioext_info[ioext_num].num_uarts] = uart;
			ioext_info[ioext_num].num_uarts++;

			rs_table[line].sw = &ioext_ser_switch;

			/* We don't use these values */
			rs_table[line].nr_uarts = 0;
			rs_table[line].board_base = board;

			init_ioext_uart(uart);
		}

		/* Now we handle the parallel port
		   (possibly as a PLIP device) */
		if (is_ioext) {
			/* Make sure that LP_PINTEN is ALWAYS set,
			   to prevent continuous interrupts */
			board->par.CTRL = LP_PINTEN|LP_PINITP|LP_PSELECP;

#if defined(CONFIG_GVPIOEXT_PLIP) || defined(CONFIG_GVPIOEXT_PLIP_MODULE)
			ioext_info[ioext_num].dev = &ioext_dev_plip[ioext_num];
			register_netdev(&ioext_dev_plip[ioext_num]);
			strcat(support_string, "/plip");
#endif
#if defined(CONFIG_GVPIOEXT_LP) || defined(CONFIG_GVPIOEXT_LP_MODULE)
			ioext_info[ioext_num].lp_table =
				&ioext_lp_table[ioext_num];
			ioext_info[ioext_num].lp_table->base =
				(void *)&ioext_info[ioext_num];
			ioext_info[ioext_num].lp_dev =
				register_parallel(ioext_info[ioext_num].lp_table, -1);
			if (ioext_info[ioext_num].lp_dev < 0) {
				printk( "Cannot register IO-Extender parallel port: no free device\n" );
			}
			strcat(support_string, "/lp");
#endif
		}

		/* Install ISR if it hasn't been installed already */
		if (!isr_installed) {
			request_irq(IRQ_AMIGA_EXTER, ioext_interrupt, 0,
				    support_string, ioext_info);
			isr_installed++;
		}
		ioext_num++;

		/*
		 * Set the board to INT6 and RTSx port-control.
		 * (GVP uses this setup).
		 */
		board->CTRL = (IRQ_SEL|PORT0_CTRL|PORT1_CTRL);

		board->CNTR |= GVP_IRQ_ENA;
	}

	return 0;
}

#ifdef MODULE
int init_module(void)
{
	return(ioext_init());
}

void cleanup_module(void)
{
  int i;

  for (i = 0; i < ioext_num; i++) {
    IOEXT_struct *board = ioext_info[i].board;
    int j;

    /* Disable board-interrupts */
    board->CNTR &= ~GVP_IRQ_ENA;

    /* Disable "master" interrupt select on uarts */
    board->uart0.MCR = 0;
    board->uart1.MCR = 0;

    /* Disable all uart interrupts */
    board->uart0.IER = 0;
    board->uart1.IER = 0;

    for (j = 0; j < ioext_info[i].num_uarts; j++) {
      unregister_serial(ioext_info[i].line[j]);
    }

#if defined(CONFIG_GVPIOEXT_PLIP) || defined(CONFIG_GVPIOEXT_PLIP_MODULE)
    if (ioext_info[i].dev != 0) {
      unregister_netdev(ioext_info[i].dev);
    }
#endif
#if defined(CONFIG_GVPIOEXT_LP) || defined(CONFIG_GVPIOEXT_LP_MODULE)
    if (ioext_info[i].lp_dev >= 0) {
      unregister_parallel(ioext_info[i].lp_dev);
    }
#endif

    release_mem_region(ZTWO_PADDR(board), sizeof(IOEXT_struct));
  }

  if (ioext_num != 0) {
    /* Indicate to the IRQ that nothing needs to be serviced */
    ioext_num = 0;

    free_irq(IRQ_AMIGA_EXTER, ioext_info);
  }
}
#endif
